import React from 'react';
import { Link } from 'react-router-dom';
import { Product } from '../types';
import { Check, ShoppingCart, Info } from 'lucide-react';
import { useCart } from '../context/CartContext';

interface ProductCardProps {
  product: Product;
}

export const ProductCard: React.FC<ProductCardProps> = ({ product }) => {
  const { addToCart } = useCart();

  const handleAddToCart = (e: React.MouseEvent) => {
    e.preventDefault();
    e.stopPropagation();
    addToCart(product);
  };

  return (
    <div className="group relative bg-white rounded-2xl shadow-sm hover:shadow-xl transition-all duration-300 border border-gray-100 flex flex-col h-full overflow-hidden">
      {/* Discount Badge */}
      <div className="absolute top-4 right-4 z-10 bg-red-500 text-white text-xs font-bold px-2 py-1 rounded-full">
        {Math.round(((product.originalPrice - product.price) / product.originalPrice) * 100)}% OFF
      </div>

      {/* Image Area */}
      <Link to={`/products/${product.id}`} className="block h-48 overflow-hidden bg-gray-100 relative">
         <img 
           src={product.image} 
           alt={product.name}
           className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-110"
         />
         <div className="absolute inset-0 bg-black/10 group-hover:bg-black/0 transition-colors" />
      </Link>

      {/* Content */}
      <div className="p-6 flex flex-col flex-1">
        <div className="mb-4">
          <p className="text-xs font-semibold text-indigo-600 uppercase tracking-wide mb-1">{product.category}</p>
          <Link to={`/products/${product.id}`}>
            <h3 className="text-xl font-bold text-gray-900 group-hover:text-indigo-600 transition-colors">
              {product.name}
            </h3>
          </Link>
          <p className="mt-2 text-sm text-gray-500 line-clamp-2">
            {product.shortDescription}
          </p>
        </div>

        {/* Features Preview */}
        <div className="mb-6 space-y-2">
          {product.features.slice(0, 3).map((feature, idx) => (
            <div key={idx} className="flex items-center text-xs text-gray-600">
              <Check className="h-3 w-3 text-green-500 mr-2 flex-shrink-0" />
              <span className="truncate">{feature}</span>
            </div>
          ))}
        </div>

        {/* Footer */}
        <div className="mt-auto flex items-end justify-between gap-4">
          <div>
            <p className="text-xs text-gray-400 line-through">${product.originalPrice.toFixed(2)}</p>
            <p className="text-2xl font-bold text-gray-900">${product.price.toFixed(2)}<span className="text-sm font-normal text-gray-500">/mo</span></p>
          </div>
          
          <button 
            onClick={handleAddToCart}
            className="p-3 bg-gray-100 hover:bg-indigo-600 text-gray-900 hover:text-white rounded-xl transition-colors duration-200"
            aria-label="Add to Cart"
          >
            <ShoppingCart className="h-5 w-5" />
          </button>
        </div>
        
        <div className="mt-4 pt-4 border-t border-gray-100 flex items-center justify-between text-xs text-gray-500">
            <span>{product.deliveryTime}</span>
            <Link to={`/products/${product.id}`} className="flex items-center hover:text-indigo-600">
                Details <Info className="ml-1 h-3 w-3" />
            </Link>
        </div>
      </div>
    </div>
  );
};