import React from 'react';
import { useParams, Link } from 'react-router-dom';
import { PRODUCTS } from '../constants';
import { Check, Star, Shield, Clock, Mail, ShoppingCart } from 'lucide-react';
import { useCart } from '../context/CartContext';

export const ProductDetail: React.FC = () => {
  const { id } = useParams<{ id: string }>();
  const product = PRODUCTS.find(p => p.id === id);
  const { addToCart } = useCart();

  if (!product) {
    return (
      <div className="min-h-[50vh] flex flex-col items-center justify-center">
        <h2 className="text-2xl font-bold text-gray-900">Product Not Found</h2>
        <Link to="/products" className="mt-4 text-indigo-600 hover:text-indigo-800">Back to Products</Link>
      </div>
    );
  }

  return (
    <div className="bg-gray-50 min-h-screen py-12">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="bg-white rounded-3xl shadow-sm border border-gray-100 overflow-hidden">
          <div className="grid grid-cols-1 md:grid-cols-2">
            
            {/* Image Section */}
            <div className="relative h-96 md:h-full bg-gray-100">
              <img 
                src={product.image} 
                alt={product.name}
                className="absolute inset-0 w-full h-full object-cover"
              />
              <div className="absolute top-4 left-4 bg-white/90 backdrop-blur px-3 py-1 rounded-full text-sm font-semibold text-gray-900">
                {product.category}
              </div>
            </div>

            {/* Content Section */}
            <div className="p-8 md:p-12 lg:p-16">
              <div className="flex items-center gap-2 mb-4">
                <div className="flex text-yellow-400">
                  {[...Array(5)].map((_, i) => (
                    <Star key={i} className={`h-5 w-5 ${i < Math.floor(product.rating) ? 'fill-current' : 'text-gray-300'}`} />
                  ))}
                </div>
                <span className="text-sm text-gray-500">({product.reviewsCount} reviews)</span>
              </div>

              <h1 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">{product.name}</h1>
              
              <div className="flex items-end gap-4 mb-8">
                <p className="text-4xl font-bold text-indigo-600">${product.price.toFixed(2)}</p>
                <div className="mb-1">
                  <p className="text-sm text-gray-500 line-through">${product.originalPrice.toFixed(2)}</p>
                  <p className="text-sm text-green-600 font-semibold">Save {Math.round(((product.originalPrice - product.price) / product.originalPrice) * 100)}%</p>
                </div>
              </div>

              <p className="text-gray-600 text-lg mb-8 leading-relaxed">
                {product.fullDescription}
              </p>

              <div className="space-y-4 mb-8">
                <h3 className="font-semibold text-gray-900">What's Included:</h3>
                <ul className="grid grid-cols-1 gap-3">
                  {product.features.map((feature, idx) => (
                    <li key={idx} className="flex items-center text-gray-700">
                      <div className="flex-shrink-0 h-6 w-6 rounded-full bg-green-100 flex items-center justify-center mr-3">
                        <Check className="h-4 w-4 text-green-600" />
                      </div>
                      {feature}
                    </li>
                  ))}
                </ul>
              </div>

              <div className="bg-gray-50 rounded-xl p-4 mb-8 border border-gray-200 grid grid-cols-2 gap-4 text-sm">
                <div className="flex items-center gap-2 text-gray-600">
                  <Clock className="h-5 w-5 text-indigo-500" />
                  <span>Instant Delivery</span>
                </div>
                <div className="flex items-center gap-2 text-gray-600">
                  <Mail className="h-5 w-5 text-indigo-500" />
                  <span>Sent via Email</span>
                </div>
                <div className="flex items-center gap-2 text-gray-600">
                  <Shield className="h-5 w-5 text-indigo-500" />
                  <span>Warranty Included</span>
                </div>
              </div>

              <button 
                onClick={() => addToCart(product)}
                className="w-full bg-indigo-600 text-white py-4 px-8 rounded-xl font-bold text-lg hover:bg-indigo-700 transition-colors shadow-lg hover:shadow-xl flex items-center justify-center gap-2"
              >
                <ShoppingCart className="h-6 w-6" /> Add to Cart
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};